dofile ("shape.lua")

local Vec3 = {}

function Vec3.new (x, y, z)
	return {
		x = x,
		y = y,
		z = z,
	}
end

local function test_AABB_to_rgn ()
	print ("AABB_to_rgn: ")
	local rgn = mcl_util.decompose_AABBs ({{
		10.0, 10.0, 10.0,
		23.0, 11.0, 25.0,
	}})
	assert (rgn.x_size == 2)
	assert (rgn.y_size == 2)
	assert (rgn.z_size == 2)
	assert (rgn.x_edges[1] == 10.0
		and rgn.x_edges[2] == 23.0)
	assert (rgn.y_edges[1] == 10.0
		and rgn.y_edges[2] == 11.0)
	assert (rgn.y_edges[1] == 10.0
		and rgn.z_edges[2] == 25.0)
	assert (rgn:is_AABB (10, 10, 10))
	assert (not rgn:is_AABB (11, 11, 11))
	assert (not rgn:is_AABB (23, 11, 25))
end

local function AABB_intersect_p (a, b)
	local x1a, y1a, z1a, x2a, y2a, z2a = unpack (a)
	local x1b, y1b, z1b, x2b, y2b, z2b = unpack (b)

	return x1a <= x1b
		and y1a <= y1b
		and z1a <= z1b
		and x2a >= x2b
		and y2a >= y2b
		and z2a >= z2b
end

local function AABB_intersect (a, b)
	local x1a, y1a, z1a, x2a, y2a, z2a = unpack (a)
	local x1b, y1b, z1b, x2b, y2b, z2b = unpack (b)
	return {
		math.max (x1a, x1b),
		math.max (y1a, y1b),
		math.max (z1a, z1b),
		math.min (x2a, x2b),
		math.min (y2a, y2b),
		math.min (z2a, z2b),
	}
end

local function any_aabb_containing (aabbs, v)
	for _, iter in pairs (aabbs) do
		local x1, y1, z1, x2, y2, z2 = unpack (iter)
		if v.x >= x1 and v.y >= y1 and v.z >= z1
		-- The extrema of AABBs are not labeled occupied.
			and v.x < x2 and v.y < y2 and v.z < z2 then
			return iter
		end
	end

	return nil
end

local function verify_union_identity_1 (region, v, should_match)
	if should_match then
		assert (region:is_AABB (v.x, v.y, v.z),
		string.format ("not region:is_AABB (%f, %f, %f)",
				       v.x, v.y, v.z))
	else
		assert (not region:is_AABB (v.x, v.y, v.z),
			string.format ("not region:is_AABB (%f, %f, %f)",
				       v.x, v.y, v.z))
	end
end

local function verify_union_identity (region, aabbs)
	local function verify_one (s1, s2, s3)
		local v = Vec3.new (s1[1], s2[2], s3[3])
		if any_aabb_containing (aabbs, v) then
			verify_union_identity_1 (region, v, true)
		else
			verify_union_identity_1 (region, v, false)
		end

		v = Vec3.new (s1[4], s2[5], s3[6])
		if any_aabb_containing (aabbs, v) then
			verify_union_identity_1 (region, v, true)
		else
			verify_union_identity_1 (region, v, false)
		end
	end

	for i, a in ipairs (aabbs) do
		for j, b in ipairs (aabbs) do
			verify_one (a, a, a)
			verify_one (a, a, b)
			verify_one (b, a, b)
			verify_one (b, a, a)
			verify_one (a, b, a)
			verify_one (b, b, a)
			verify_one (b, b, b)
		end
	end
end

local function test_AABBs_to_rgn (void)
	local test_AABBs = {
		{
			10.0, 10.0, 10.0,
			23.0, 50.0, 25.0,
		},
		{
			41.0, 43.0, 50.0,
			45.0, 60.0, 55.0,
		},
		{
			100.0, 114.0, 78.0,
			200.0, 300.0, 224.0,
		},
		{
			100.0, 100.0, 100.0,
			200.0, 200.0, 200.0,
		},
		{
			-8.0, -8.0, -8.0,
			8.0, 8.0, 8.0,
		},
		{
			-7.5, 16.0, -7.5,
			8.0, 24.0, 8.0,
		},
		{
			-7.0, 32.0, -7.0,
			8.0, 40.0, 8.0,
		},
	}

	print ("AABBs_to_rgn: ")
	local rgn = mcl_util.decompose_AABBs (test_AABBs)
	assert (rgn:is_AABB (10, 10, 10))
	verify_union_identity (rgn, test_AABBs)
	assert (rgn:is_AABB (10, 10, 10))
	assert (rgn:is_AABB (41, 43, 50))
	assert (rgn:is_AABB (100, 100, 100))
end

local function verify_intersection_identity_1 (region, l, r)
	-- Compute the intersection of ITER with every AABB in L.
	for _, iter in ipairs (l) do
		for _, rhs in ipairs (r) do
			if AABB_intersect_p (iter, rhs) then
				-- Verify that the top-left corner of
				-- the AABB is a valid AABB.
				local test = AABB_intersect (iter, rhs)
				local v = Vec3.new (test[1], test[2], test[3])
				verify_union_identity_1 (region, v, true)
			end
		end
	end
end

local function verify_intersection_identity (region, l, r)
	verify_intersection_identity_1 (region, l, r)
	verify_intersection_identity_1 (region, r, l)
end

local function test_region_ops ()
	local basic_cube = {
		-8.0, -8.0, -8.0,
		8.0, 8.0, 8.0,
	}
	local larger_cube = {
		-16.0, -16.0, -16.0,
		16.0, 16.0, 16.0,
	}

	print ("region_ops: ")
	local a = mcl_util.decompose_AABBs ({basic_cube})
	local b = mcl_util.decompose_AABBs ({basic_cube})
	assert (a:equal_p (b))
	b = mcl_util.decompose_AABBs ({larger_cube})
	assert (not a:equal_p (b))

	local staggered = {
		{
			-8.0, -8.0, -8.0,
			8.0, 8.0, 8.0,
		},
		{
			-7.5, 16.0, -7.5,
			8.0, 24.0, 8.0,
		},
		{
			-7.0, 32.0, -7.0,
			8.0, 40.0, 8.0,
		},
	}

	local staggered_1 = {
		{
			-8.0, -8.0, -8.0,
			4.0, 8.0, 4.0,
		},
		{
			-7.5, 16.0, -7.5,
			4.0, 24.0, 4.0,
		},
		{
			-7.0, 32.0, -7.0,
			4.0, 40.0, 4.0,
		},
	}

	a = mcl_util.decompose_AABBs (staggered)
	b = mcl_util.decompose_AABBs (staggered)
	assert (a:equal_p (b))
	a = mcl_util.decompose_AABBs (staggered)
	b = mcl_util.decompose_AABBs (staggered_1)
	assert (not a:equal_p (b))

	a = mcl_util.decompose_AABBs ({larger_cube})
	b = mcl_util.decompose_AABBs ({basic_cube})
	local dest = a:op (b, mcl_util.OP_AND)
	assert (dest:equal_p (b))
	assert (not dest:equal_p (a))
	dest = a:op (b, mcl_util.OP_OR)
	assert (dest:equal_p (a))
	assert (not dest:equal_p (b))

	a = mcl_util.decompose_AABBs (staggered)
	b = mcl_util.decompose_AABBs (staggered_1)
	dest = a:op (b, mcl_util.OP_AND) -- staggered_1 & staggered
	assert (dest:equal_p (b))
	assert (not dest:equal_p (a))
	verify_intersection_identity (dest, staggered, staggered_1)
	dest = a:op (b, mcl_util.OP_OR) -- staggered_1 | staggered
	assert (dest:equal_p (a))
	assert (not dest:equal_p (b))

	local big_cube = {
		-8.0, -8.0, -8.0,
		8.0, 8.0, 8.0,
	}
	local small_cube = {
		0.0, 0.0, 0.0,
		8.0, 8.0, 8.0,
	}
	local expected = {
		{
			-8.0, -8.0, -8.0,
			8.0, 0.0, 8.0,
		},
		{
			-8.0, 0.0, -8.0,
			0.0, 8.0, 8.0,
		},
		{
			-8.0, 0.0, -8.0,
			8.0, 8.0, 0.0,
		},
	}

	a = mcl_util.decompose_AABBs ({big_cube})
	b = mcl_util.decompose_AABBs ({small_cube})
	dest = a:op (b, mcl_util.OP_SUB)
	a = mcl_util.decompose_AABBs (expected)
	assert (dest:equal_p (a))
	assert (not dest:equal_p (b))
	a = dest:op (dest, mcl_util.OP_SUB)
	assert (a:is_empty ())
end

local big_region = {
  {
    14.00, 6.00, 37.00, 54.00, 9.00, 40.00,
  },
  {
    -10.00, -10.00, -47.00, 6.00, -8.00, -10.00,
  },
  {
    -45.00, -44.00, 10.00, -5.00, -20.00, 53.00,
  },
  {
    45.00, 12.00, -19.00, 67.00, 52.00, 12.00,
  },
  {
    -17.00, 35.00, -41.00, 26.00, 84.00, -32.00,
  },
  {
    -29.00, 31.00, 46.00, 13.00, 51.00, 73.00,
  },
  {
    -34.00, -32.00, -45.00, 3.00, -25.00, 3.00,
  },
  {
    -47.00, -10.00, 16.00, 2.00, 7.00, 37.00,
  },
  {
    -45.00, 3.00, 47.00, -33.00, 10.00, 48.00,
  },
  {
    -6.00, -13.00, -48.00, 20.00, 24.00, -21.00,
  },
  {
    -28.00, 10.00, -50.00, 15.00, 60.00, -24.00,
  },
  {
    -25.00, -8.00, 39.00, 10.00, 13.00, 56.00,
  },
  {
    -48.00, 11.00, -28.00, -36.00, 29.00, 5.00,
  },
  {
    34.00, 36.00, 42.00, 35.00, 83.00, 88.00,
  },
  {
    10.00, -40.00, -34.00, 44.00, -24.00, -6.00,
  },
  {
    2.00, 48.00, -20.00, 31.00, 61.00, -2.00,
  },
  {
    20.00, 37.00, -11.00, 29.00, 81.00, 1.00,
  },
  {
    -18.00, -44.00, 14.00, 28.00, -39.00, 48.00,
  },
  {
    -19.00, 15.00, 11.00, -2.00, 17.00, 31.00,
  },
  {
    9.00, -42.00, 17.00, 34.00, -33.00, 27.00,
  },
  {
    11.00, 35.00, 30.00, 35.00, 73.00, 42.00,
  },
  {
    -7.00, 16.00, -26.00, 0.00, 17.00, 2.00,
  },
  {
    14.00, 44.00, 32.00, 30.00, 46.00, 60.00,
  },
  {
    36.00, -31.00, -46.00, 44.00, -19.00, 1.00,
  },
  {
    -48.00, 39.00, 43.00, -20.00, 71.00, 92.00,
  },
  {
    13.00, -21.00, 49.00, 43.00, -2.00, 61.00,
  },
  {
    2.00, -12.00, -13.00, 39.00, -7.00, 26.00,
  },
  {
    37.00, -14.00, 29.00, 57.00, -2.00, 38.00,
  },
  {
    -15.00, -42.00, 44.00, -11.00, -4.00, 82.00,
  },
  {
    27.00, -30.00, -8.00, 67.00, 0.00, 18.00,
  },
  {
    -39.00, -22.00, 43.00, -17.00, -20.00, 59.00,
  },
  {
    -11.00, 20.00, -39.00, 13.00, 51.00, -22.00,
  },
  {
    -10.00, 14.00, -2.00, 6.00, 16.00, 9.00,
  },
  {
    43.00, -41.00, -21.00, 91.00, -11.00, -18.00,
  },
  {
    48.00, -4.00, 31.00, 80.00, 26.00, 66.00,
  },
  {
    -20.00, 7.00, -48.00, -14.00, 28.00, -15.00,
  },
  {
    -3.00, -41.00, -20.00, 22.00, 2.00, 29.00,
  },
  {
    26.00, -40.00, -15.00, 72.00, -6.00, 8.00,
  },
  {
    9.00, 26.00, -25.00, 47.00, 34.00, -3.00,
  },
  {
    -19.00, 23.00, 33.00, -5.00, 65.00, 37.00,
  },
  {
    35.00, 39.00, 19.00, 36.00, 67.00, 67.00,
  },
  {
    -29.00, -3.00, -22.00, -26.00, -2.00, 11.00,
  },
  {
    20.00, 16.00, -41.00, 22.00, 25.00, -35.00,
  },
  {
    37.00, 38.00, -30.00, 74.00, 61.00, 4.00,
  },
  {
    -29.00, 48.00, 39.00, -7.00, 85.00, 43.00,
  },
  {
    -31.00, 5.00, -13.00, 18.00, 54.00, -11.00,
  },
  {
    48.00, -39.00, 45.00, 98.00, 9.00, 54.00,
  },
  {
    46.00, 11.00, -41.00, 94.00, 13.00, -24.00,
  },
  {
    2.00, 7.00, 38.00, 38.00, 26.00, 81.00,
  },
  {
    38.00, 6.00, 32.00, 47.00, 7.00, 56.00,
  },
  {
    26.00, 28.00, -10.00, 46.00, 32.00, 3.00,
  },
  {
    -1.00, 22.00, -20.00, 45.00, 48.00, 1.00,
  },
  {
    -7.00, 26.00, 10.00, 35.00, 72.00, 48.00,
  },
  {
    -6.00, 48.00, -1.00, 23.00, 74.00, 45.00,
  },
  {
    4.00, 20.00, -6.00, 23.00, 41.00, 0.00,
  },
  {
    42.00, -31.00, 24.00, 86.00, -13.00, 57.00,
  },
  {
    11.00, -44.00, -4.00, 21.00, -36.00, 23.00,
  },
  {
    -37.00, -37.00, 4.00, 1.00, -15.00, 50.00,
  },
  {
    13.00, 43.00, 22.00, 33.00, 90.00, 27.00,
  },
  {
    49.00, 37.00, 1.00, 53.00, 47.00, 16.00,
  },
  {
    2.00, 18.00, 48.00, 35.00, 35.00, 77.00,
  },
  {
    34.00, 37.00, 26.00, 64.00, 81.00, 29.00,
  },
  {
    -31.00, 17.00, 34.00, -15.00, 63.00, 36.00,
  },
  {
    -10.00, -14.00, -33.00, -7.00, 18.00, 5.00,
  },
  {
    26.00, 9.00, -17.00, 33.00, 43.00, 23.00,
  },
  {
    29.00, 24.00, -10.00, 72.00, 44.00, 16.00,
  },
  {
    41.00, 39.00, 21.00, 73.00, 81.00, 31.00,
  },
  {
    -21.00, -49.00, 10.00, 10.00, -36.00, 37.00,
  },
  {
    -12.00, 41.00, 26.00, 25.00, 67.00, 33.00,
  },
  {
    -27.00, -39.00, -40.00, -4.00, 3.00, -15.00,
  },
  {
    28.00, 35.00, -14.00, 74.00, 76.00, 28.00,
  },
  {
    5.00, -15.00, -23.00, 43.00, -12.00, -3.00,
  },
  {
    -47.00, 31.00, 24.00, -36.00, 47.00, 51.00,
  },
  {
    -47.00, 26.00, -43.00, -41.00, 27.00, -41.00,
  },
  {
    34.00, -32.00, 47.00, 62.00, -16.00, 48.00,
  },
  {
    -19.00, 38.00, 26.00, -15.00, 44.00, 59.00,
  },
  {
    -36.00, 42.00, -45.00, -15.00, 45.00, -39.00,
  },
  {
    -39.00, 22.00, 4.00, -4.00, 24.00, 22.00,
  },
  {
    25.00, 38.00, -11.00, 61.00, 42.00, 25.00,
  },
  {
    -42.00, 3.00, 22.00, -25.00, 27.00, 49.00,
  },
  {
    39.00, -24.00, 29.00, 49.00, 2.00, 41.00,
  },
  {
    -33.00, -47.00, -24.00, 12.00, 1.00, 14.00,
  },
  {
    2.00, -30.00, -46.00, 11.00, 2.00, -37.00,
  },
  {
    14.00, -21.00, -29.00, 46.00, 19.00, 13.00,
  },
  {
    -30.00, -50.00, 1.00, -28.00, -43.00, 35.00,
  },
  {
    33.00, 21.00, -22.00, 57.00, 63.00, -10.00,
  },
  {
    -7.00, 40.00, -10.00, 42.00, 57.00, 23.00,
  },
  {
    6.00, 27.00, -23.00, 44.00, 45.00, -21.00,
  },
  {
    30.00, -20.00, -12.00, 49.00, 26.00, -2.00,
  },
  {
    23.00, 8.00, -9.00, 36.00, 24.00, -3.00,
  },
  {
    -4.00, -39.00, -30.00, 21.00, 10.00, 12.00,
  },
  {
    -1.00, -40.00, 37.00, 24.00, -26.00, 40.00,
  },
  {
    17.00, 36.00, -45.00, 52.00, 60.00, -15.00,
  },
  {
    -41.00, -2.00, -35.00, -12.00, 46.00, -29.00,
  },
  {
    -3.00, -34.00, -36.00, 23.00, -26.00, -19.00,
  },
  {
    -48.00, 36.00, 36.00, -45.00, 67.00, 77.00,
  },
}

local big_region_1 = {
  {
    -25.00, 22.00, 46.00, 20.00, 23.00, 49.00,
  },
  {
    46.00, -20.00, -5.00, 62.00, 10.00, 23.00,
  },
  {
    47.00, 0.00, -8.00, 89.00, 40.00, 42.00,
  },
  {
    -41.00, 16.00, 32.00, -9.00, 32.00, 67.00,
  },
  {
    -20.00, 29.00, -46.00, -4.00, 32.00, -4.00,
  },
  {
    23.00, -25.00, 11.00, 30.00, -22.00, 36.00,
  },
  {
    3.00, 43.00, -8.00, 39.00, 62.00, 13.00,
  },
  {
    11.00, -14.00, 46.00, 43.00, -3.00, 55.00,
  },
  {
    -29.00, -39.00, 5.00, -5.00, -23.00, 51.00,
  },
  {
    -35.00, 47.00, 5.00, -2.00, 56.00, 51.00,
  },
  {
    -49.00, 15.00, -31.00, -4.00, 19.00, 1.00,
  },
  {
    0.00, 40.00, 42.00, 21.00, 80.00, 64.00,
  },
  {
    23.00, 8.00, 45.00, 30.00, 30.00, 63.00,
  },
  {
    35.00, -12.00, 20.00, 56.00, 17.00, 42.00,
  },
  {
    41.00, 17.00, -24.00, 55.00, 67.00, 13.00,
  },
  {
    34.00, -31.00, -24.00, 74.00, 15.00, -10.00,
  },
  {
    -16.00, -32.00, 32.00, 29.00, 13.00, 80.00,
  },
  {
    -37.00, 23.00, 49.00, -23.00, 26.00, 67.00,
  },
  {
    -49.00, -32.00, -7.00, -6.00, 7.00, 33.00,
  },
  {
    36.00, -48.00, 20.00, 57.00, -4.00, 43.00,
  },
  {
    38.00, 36.00, -43.00, 87.00, 84.00, -36.00,
  },
  {
    8.00, 44.00, -38.00, 14.00, 61.00, -32.00,
  },
  {
    37.00, -24.00, -1.00, 84.00, 23.00, 0.00,
  },
  {
    -25.00, -24.00, -23.00, 1.00, 13.00, -14.00,
  },
  {
    -30.00, 49.00, 34.00, -27.00, 76.00, 72.00,
  },
  {
    10.00, 13.00, 24.00, 21.00, 36.00, 30.00,
  },
  {
    -36.00, 11.00, 7.00, -4.00, 41.00, 40.00,
  },
  {
    -27.00, -21.00, 41.00, -17.00, -15.00, 52.00,
  },
  {
    0.00, -12.00, -42.00, 22.00, 3.00, -31.00,
  },
  {
    3.00, 14.00, 32.00, 44.00, 61.00, 51.00,
  },
  {
    -7.00, -7.00, 28.00, 11.00, 42.00, 56.00,
  },
  {
    -23.00, 47.00, 28.00, 27.00, 68.00, 62.00,
  },
  {
    -28.00, -30.00, -6.00, 4.00, -1.00, 12.00,
  },
  {
    -14.00, -15.00, -30.00, 34.00, 29.00, -14.00,
  },
  {
    -20.00, -8.00, -33.00, 24.00, 16.00, 9.00,
  },
  {
    -12.00, 49.00, -36.00, -2.00, 90.00, -21.00,
  },
  {
    -18.00, -29.00, -30.00, 21.00, -19.00, -9.00,
  },
  {
    14.00, -19.00, -33.00, 30.00, 4.00, -13.00,
  },
  {
    -4.00, 46.00, -33.00, -3.00, 51.00, -11.00,
  },
  {
    32.00, -42.00, -9.00, 76.00, -41.00, -8.00,
  },
  {
    17.00, 45.00, 8.00, 51.00, 77.00, 9.00,
  },
  {
    -12.00, -10.00, 9.00, 3.00, -2.00, 12.00,
  },
  {
    37.00, -50.00, 14.00, 72.00, -8.00, 23.00,
  },
  {
    -32.00, 27.00, 9.00, 10.00, 64.00, 54.00,
  },
  {
    -9.00, -39.00, 7.00, 13.00, -13.00, 27.00,
  },
  {
    0.00, 24.00, 32.00, 13.00, 33.00, 56.00,
  },
  {
    -19.00, -39.00, 4.00, -18.00, -2.00, 21.00,
  },
  {
    -18.00, -46.00, -43.00, 13.00, -41.00, -18.00,
  },
  {
    38.00, -18.00, 5.00, 82.00, 19.00, 29.00,
  },
  {
    33.00, -3.00, -40.00, 76.00, 2.00, 6.00,
  },
  {
    7.00, 0.00, 8.00, 14.00, 35.00, 16.00,
  },
  {
    35.00, 5.00, 12.00, 68.00, 41.00, 32.00,
  },
  {
    -46.00, -9.00, 43.00, -42.00, 38.00, 70.00,
  },
  {
    38.00, 33.00, 34.00, 78.00, 74.00, 75.00,
  },
  {
    -47.00, -30.00, 45.00, -31.00, 20.00, 71.00,
  },
  {
    0.00, 22.00, 43.00, 40.00, 52.00, 67.00,
  },
  {
    -28.00, 44.00, 34.00, 7.00, 61.00, 56.00,
  },
  {
    48.00, 38.00, -41.00, 91.00, 85.00, -20.00,
  },
  {
    46.00, -45.00, 27.00, 70.00, -30.00, 70.00,
  },
  {
    48.00, -21.00, -7.00, 65.00, -10.00, 33.00,
  },
  {
    -33.00, 21.00, 16.00, -24.00, 55.00, 50.00,
  },
  {
    46.00, -34.00, -41.00, 63.00, 8.00, -2.00,
  },
  {
    -48.00, -2.00, 14.00, -23.00, 36.00, 28.00,
  },
  {
    41.00, 8.00, -3.00, 64.00, 32.00, 31.00,
  },
  {
    -5.00, 13.00, -36.00, 36.00, 19.00, 2.00,
  },
  {
    -9.00, -9.00, 39.00, 9.00, 29.00, 52.00,
  },
  {
    -31.00, -1.00, 46.00, -7.00, 47.00, 62.00,
  },
  {
    13.00, -20.00, -15.00, 17.00, 4.00, 18.00,
  },
  {
    26.00, -23.00, 31.00, 27.00, 11.00, 40.00,
  },
  {
    -15.00, 9.00, -10.00, 22.00, 36.00, 38.00,
  },
  {
    -5.00, 36.00, 4.00, 11.00, 70.00, 11.00,
  },
  {
    13.00, -32.00, 13.00, 52.00, 0.00, 35.00,
  },
  {
    -29.00, 45.00, -25.00, 7.00, 51.00, 10.00,
  },
  {
    -7.00, -31.00, 6.00, 35.00, -22.00, 56.00,
  },
  {
    13.00, 11.00, 22.00, 39.00, 42.00, 43.00,
  },
  {
    34.00, -49.00, -26.00, 65.00, -9.00, -20.00,
  },
  {
    9.00, 20.00, 28.00, 35.00, 70.00, 45.00,
  },
  {
    -34.00, -36.00, 10.00, -17.00, -10.00, 38.00,
  },
  {
    -18.00, 16.00, 20.00, 27.00, 28.00, 45.00,
  },
  {
    32.00, -42.00, -8.00, 77.00, -5.00, 34.00,
  },
  {
    26.00, 12.00, 7.00, 62.00, 47.00, 25.00,
  },
  {
    8.00, 42.00, -23.00, 41.00, 85.00, 17.00,
  },
  {
    -17.00, -49.00, 28.00, -9.00, -38.00, 72.00,
  },
  {
    23.00, -9.00, -2.00, 36.00, 36.00, 36.00,
  },
  {
    10.00, 19.00, 31.00, 45.00, 49.00, 69.00,
  },
  {
    38.00, -28.00, -33.00, 59.00, -5.00, -13.00,
  },
  {
    36.00, 17.00, -7.00, 60.00, 23.00, 42.00,
  },
  {
    -50.00, -10.00, -17.00, -25.00, -5.00, 10.00,
  },
  {
    14.00, -15.00, 8.00, 19.00, 4.00, 55.00,
  },
  {
    -22.00, -8.00, 29.00, 7.00, 32.00, 45.00,
  },
  {
    -49.00, 24.00, 7.00, -31.00, 43.00, 57.00,
  },
  {
    -8.00, 10.00, -18.00, 32.00, 28.00, -3.00,
  },
  {
    38.00, 8.00, -21.00, 81.00, 54.00, 17.00,
  },
  {
    10.00, 0.00, 21.00, 26.00, 34.00, 35.00,
  },
  {
    -15.00, 25.00, 39.00, 30.00, 45.00, 43.00,
  },
  {
    30.00, 5.00, -49.00, 76.00, 18.00, -3.00,
  },
  {
    -4.00, -6.00, 22.00, 11.00, 30.00, 33.00,
  },
  {
    12.00, 14.00, -31.00, 58.00, 55.00, 7.00,
  },
  {
    44.00, -14.00, 2.00, 67.00, 32.00, 13.00,
  },
  {
    42.00, -34.00, 47.00, 74.00, -3.00, 90.00,
  },
  {
    -14.00, 6.00, -25.00, 17.00, 53.00, 24.00,
  },
  {
    -19.00, -27.00, 14.00, 4.00, 1.00, 59.00,
  },
  {
    -21.00, 19.00, -41.00, 20.00, 24.00, -36.00,
  },
  {
    21.00, 4.00, 8.00, 41.00, 37.00, 11.00,
  },
  {
    0.00, 23.00, -1.00, 1.00, 51.00, 25.00,
  },
  {
    41.00, -8.00, -47.00, 48.00, 8.00, -39.00,
  },
  {
    -10.00, 48.00, 27.00, 0.00, 66.00, 28.00,
  },
  {
    -6.00, 21.00, -9.00, 26.00, 69.00, 40.00,
  },
  {
    34.00, 9.00, 24.00, 36.00, 58.00, 46.00,
  },
  {
    -9.00, 39.00, 33.00, -1.00, 60.00, 62.00,
  },
  {
    -11.00, 29.00, -5.00, 31.00, 59.00, 33.00,
  },
  {
    19.00, -50.00, 38.00, 33.00, -40.00, 54.00,
  },
  {
    -13.00, -21.00, -41.00, 5.00, 0.00, 9.00,
  },
  {
    41.00, -26.00, 47.00, 77.00, 0.00, 53.00,
  },
  {
    24.00, 33.00, 11.00, 74.00, 83.00, 35.00,
  },
  {
    19.00, 24.00, 44.00, 41.00, 49.00, 55.00,
  },
  {
    2.00, -7.00, 12.00, 8.00, 10.00, 48.00,
  },
  {
    49.00, -35.00, 0.00, 63.00, -2.00, 29.00,
  },
  {
    -31.00, 45.00, 31.00, -12.00, 46.00, 59.00,
  },
  {
    21.00, -11.00, -16.00, 54.00, 34.00, 19.00,
  },
  {
    -19.00, -33.00, 45.00, 14.00, -9.00, 54.00,
  },
  {
    1.00, 22.00, 2.00, 46.00, 69.00, 46.00,
  },
  {
    -41.00, -35.00, -45.00, -11.00, -1.00, -32.00,
  },
  {
    -35.00, -10.00, -35.00, 3.00, 22.00, 11.00,
  },
  {
    -44.00, 14.00, 22.00, -33.00, 22.00, 31.00,
  },
  {
    -32.00, -43.00, 3.00, 8.00, 5.00, 45.00,
  },
  {
    49.00, 3.00, 32.00, 68.00, 24.00, 62.00,
  },
  {
    -14.00, 10.00, -26.00, 34.00, 27.00, 22.00,
  },
  {
    -50.00, 30.00, 47.00, -8.00, 45.00, 59.00,
  },
  {
    4.00, 26.00, 22.00, 44.00, 44.00, 47.00,
  },
  {
    -41.00, -24.00, -5.00, -9.00, -21.00, 18.00,
  },
  {
    28.00, -30.00, 18.00, 67.00, -6.00, 29.00,
  },
  {
    -20.00, -28.00, -25.00, 27.00, 22.00, 17.00,
  },
  {
    -41.00, 46.00, -17.00, -25.00, 90.00, 28.00,
  },
  {
    21.00, 23.00, 39.00, 59.00, 66.00, 67.00,
  },
  {
    36.00, -2.00, -35.00, 80.00, 2.00, -19.00,
  },
  {
    37.00, -36.00, -41.00, 77.00, -9.00, 4.00,
  },
  {
    1.00, 36.00, -5.00, 21.00, 49.00, -2.00,
  },
  {
    -16.00, -12.00, 7.00, 18.00, 15.00, 21.00,
  },
  {
    -21.00, 15.00, -2.00, 0.00, 46.00, 41.00,
  },
  {
    33.00, -43.00, 4.00, 65.00, -34.00, 14.00,
  },
  {
    4.00, -2.00, -41.00, 14.00, 37.00, 0.00,
  },
  {
    1.00, -21.00, 5.00, 41.00, 24.00, 28.00,
  },
  {
    34.00, -21.00, 35.00, 73.00, -19.00, 62.00,
  },
  {
    -25.00, 22.00, 46.00, 20.00, 23.00, 49.00,
  },
  {
    46.00, -20.00, -5.00, 62.00, 10.00, 23.00,
  },
  {
    47.00, 0.00, -8.00, 89.00, 40.00, 42.00,
  },
  {
    -41.00, 16.00, 32.00, -9.00, 32.00, 67.00,
  },
  {
    -20.00, 29.00, -46.00, -4.00, 32.00, -4.00,
  },
  {
    23.00, -25.00, 11.00, 30.00, -22.00, 36.00,
  },
  {
    3.00, 43.00, -8.00, 39.00, 62.00, 13.00,
  },
  {
    11.00, -14.00, 46.00, 43.00, -3.00, 55.00,
  },
  {
    -29.00, -39.00, 5.00, -5.00, -23.00, 51.00,
  },
  {
    -35.00, 47.00, 5.00, -2.00, 56.00, 51.00,
  },
  {
    -49.00, 15.00, -31.00, -4.00, 19.00, 1.00,
  },
  {
    0.00, 40.00, 42.00, 21.00, 80.00, 64.00,
  },
  {
    23.00, 8.00, 45.00, 30.00, 30.00, 63.00,
  },
  {
    35.00, -12.00, 20.00, 56.00, 17.00, 42.00,
  },
  {
    41.00, 17.00, -24.00, 55.00, 67.00, 13.00,
  },
  {
    34.00, -31.00, -24.00, 74.00, 15.00, -10.00,
  },
  {
    -16.00, -32.00, 32.00, 29.00, 13.00, 80.00,
  },
  {
    -37.00, 23.00, 49.00, -23.00, 26.00, 67.00,
  },
  {
    -49.00, -32.00, -7.00, -6.00, 7.00, 33.00,
  },
  {
    36.00, -48.00, 20.00, 57.00, -4.00, 43.00,
  },
  {
    38.00, 36.00, -43.00, 87.00, 84.00, -36.00,
  },
  {
    8.00, 44.00, -38.00, 14.00, 61.00, -32.00,
  },
  {
    37.00, -24.00, -1.00, 84.00, 23.00, 0.00,
  },
  {
    -25.00, -24.00, -23.00, 1.00, 13.00, -14.00,
  },
  {
    -30.00, 49.00, 34.00, -27.00, 76.00, 72.00,
  },
  {
    10.00, 13.00, 24.00, 21.00, 36.00, 30.00,
  },
  {
    -36.00, 11.00, 7.00, -4.00, 41.00, 40.00,
  },
  {
    -27.00, -21.00, 41.00, -17.00, -15.00, 52.00,
  },
  {
    0.00, -12.00, -42.00, 22.00, 3.00, -31.00,
  },
  {
    3.00, 14.00, 32.00, 44.00, 61.00, 51.00,
  },
  {
    -7.00, -7.00, 28.00, 11.00, 42.00, 56.00,
  },
  {
    -23.00, 47.00, 28.00, 27.00, 68.00, 62.00,
  },
  {
    -28.00, -30.00, -6.00, 4.00, -1.00, 12.00,
  },
  {
    -14.00, -15.00, -30.00, 34.00, 29.00, -14.00,
  },
  {
    -20.00, -8.00, -33.00, 24.00, 16.00, 9.00,
  },
  {
    -12.00, 49.00, -36.00, -2.00, 90.00, -21.00,
  },
  {
    -18.00, -29.00, -30.00, 21.00, -19.00, -9.00,
  },
  {
    14.00, -19.00, -33.00, 30.00, 4.00, -13.00,
  },
  {
    -4.00, 46.00, -33.00, -3.00, 51.00, -11.00,
  },
  {
    32.00, -42.00, -9.00, 76.00, -41.00, -8.00,
  },
  {
    17.00, 45.00, 8.00, 51.00, 77.00, 9.00,
  },
  {
    -12.00, -10.00, 9.00, 3.00, -2.00, 12.00,
  },
  {
    37.00, -50.00, 14.00, 72.00, -8.00, 23.00,
  },
  {
    -32.00, 27.00, 9.00, 10.00, 64.00, 54.00,
  },
  {
    -9.00, -39.00, 7.00, 13.00, -13.00, 27.00,
  },
  {
    0.00, 24.00, 32.00, 13.00, 33.00, 56.00,
  },
  {
    -19.00, -39.00, 4.00, -18.00, -2.00, 21.00,
  },
  {
    -18.00, -46.00, -43.00, 13.00, -41.00, -18.00,
  },
  {
    38.00, -18.00, 5.00, 82.00, 19.00, 29.00,
  },
  {
    33.00, -3.00, -40.00, 76.00, 2.00, 6.00,
  },
  {
    7.00, 0.00, 8.00, 14.00, 35.00, 16.00,
  },
  {
    35.00, 5.00, 12.00, 68.00, 41.00, 32.00,
  },
  {
    -46.00, -9.00, 43.00, -42.00, 38.00, 70.00,
  },
  {
    38.00, 33.00, 34.00, 78.00, 74.00, 75.00,
  },
  {
    -47.00, -30.00, 45.00, -31.00, 20.00, 71.00,
  },
  {
    0.00, 22.00, 43.00, 40.00, 52.00, 67.00,
  },
  {
    -28.00, 44.00, 34.00, 7.00, 61.00, 56.00,
  },
  {
    48.00, 38.00, -41.00, 91.00, 85.00, -20.00,
  },
  {
    46.00, -45.00, 27.00, 70.00, -30.00, 70.00,
  },
  {
    48.00, -21.00, -7.00, 65.00, -10.00, 33.00,
  },
  {
    -33.00, 21.00, 16.00, -24.00, 55.00, 50.00,
  },
  {
    46.00, -34.00, -41.00, 63.00, 8.00, -2.00,
  },
  {
    -48.00, -2.00, 14.00, -23.00, 36.00, 28.00,
  },
  {
    41.00, 8.00, -3.00, 64.00, 32.00, 31.00,
  },
  {
    -5.00, 13.00, -36.00, 36.00, 19.00, 2.00,
  },
  {
    -9.00, -9.00, 39.00, 9.00, 29.00, 52.00,
  },
  {
    -31.00, -1.00, 46.00, -7.00, 47.00, 62.00,
  },
  {
    13.00, -20.00, -15.00, 17.00, 4.00, 18.00,
  },
  {
    26.00, -23.00, 31.00, 27.00, 11.00, 40.00,
  },
  {
    -15.00, 9.00, -10.00, 22.00, 36.00, 38.00,
  },
  {
    -5.00, 36.00, 4.00, 11.00, 70.00, 11.00,
  },
  {
    13.00, -32.00, 13.00, 52.00, 0.00, 35.00,
  },
  {
    -29.00, 45.00, -25.00, 7.00, 51.00, 10.00,
  },
  {
    -7.00, -31.00, 6.00, 35.00, -22.00, 56.00,
  },
  {
    13.00, 11.00, 22.00, 39.00, 42.00, 43.00,
  },
  {
    34.00, -49.00, -26.00, 65.00, -9.00, -20.00,
  },
  {
    9.00, 20.00, 28.00, 35.00, 70.00, 45.00,
  },
  {
    -34.00, -36.00, 10.00, -17.00, -10.00, 38.00,
  },
  {
    -18.00, 16.00, 20.00, 27.00, 28.00, 45.00,
  },
  {
    32.00, -42.00, -8.00, 77.00, -5.00, 34.00,
  },
  {
    26.00, 12.00, 7.00, 62.00, 47.00, 25.00,
  },
  {
    8.00, 42.00, -23.00, 41.00, 85.00, 17.00,
  },
  {
    -17.00, -49.00, 28.00, -9.00, -38.00, 72.00,
  },
  {
    23.00, -9.00, -2.00, 36.00, 36.00, 36.00,
  },
  {
    10.00, 19.00, 31.00, 45.00, 49.00, 69.00,
  },
  {
    38.00, -28.00, -33.00, 59.00, -5.00, -13.00,
  },
  {
    36.00, 17.00, -7.00, 60.00, 23.00, 42.00,
  },
  {
    -50.00, -10.00, -17.00, -25.00, -5.00, 10.00,
  },
  {
    14.00, -15.00, 8.00, 19.00, 4.00, 55.00,
  },
  {
    -22.00, -8.00, 29.00, 7.00, 32.00, 45.00,
  },
  {
    -49.00, 24.00, 7.00, -31.00, 43.00, 57.00,
  },
  {
    -8.00, 10.00, -18.00, 32.00, 28.00, -3.00,
  },
  {
    38.00, 8.00, -21.00, 81.00, 54.00, 17.00,
  },
  {
    10.00, 0.00, 21.00, 26.00, 34.00, 35.00,
  },
  {
    -15.00, 25.00, 39.00, 30.00, 45.00, 43.00,
  },
  {
    30.00, 5.00, -49.00, 76.00, 18.00, -3.00,
  },
  {
    -4.00, -6.00, 22.00, 11.00, 30.00, 33.00,
  },
  {
    12.00, 14.00, -31.00, 58.00, 55.00, 7.00,
  },
  {
    44.00, -14.00, 2.00, 67.00, 32.00, 13.00,
  },
  {
    42.00, -34.00, 47.00, 74.00, -3.00, 90.00,
  },
  {
    -14.00, 6.00, -25.00, 17.00, 53.00, 24.00,
  },
  {
    -19.00, -27.00, 14.00, 4.00, 1.00, 59.00,
  },
  {
    -21.00, 19.00, -41.00, 20.00, 24.00, -36.00,
  },
  {
    21.00, 4.00, 8.00, 41.00, 37.00, 11.00,
  },
  {
    0.00, 23.00, -1.00, 1.00, 51.00, 25.00,
  },
  {
    41.00, -8.00, -47.00, 48.00, 8.00, -39.00,
  },
  {
    -10.00, 48.00, 27.00, 0.00, 66.00, 28.00,
  },
  {
    -6.00, 21.00, -9.00, 26.00, 69.00, 40.00,
  },
  {
    34.00, 9.00, 24.00, 36.00, 58.00, 46.00,
  },
  {
    -9.00, 39.00, 33.00, -1.00, 60.00, 62.00,
  },
  {
    -11.00, 29.00, -5.00, 31.00, 59.00, 33.00,
  },
  {
    19.00, -50.00, 38.00, 33.00, -40.00, 54.00,
  },
  {
    -13.00, -21.00, -41.00, 5.00, 0.00, 9.00,
  },
  {
    41.00, -26.00, 47.00, 77.00, 0.00, 53.00,
  },
  {
    24.00, 33.00, 11.00, 74.00, 83.00, 35.00,
  },
  {
    19.00, 24.00, 44.00, 41.00, 49.00, 55.00,
  },
  {
    2.00, -7.00, 12.00, 8.00, 10.00, 48.00,
  },
  {
    49.00, -35.00, 0.00, 63.00, -2.00, 29.00,
  },
  {
    -31.00, 45.00, 31.00, -12.00, 46.00, 59.00,
  },
  {
    21.00, -11.00, -16.00, 54.00, 34.00, 19.00,
  },
  {
    -19.00, -33.00, 45.00, 14.00, -9.00, 54.00,
  },
  {
    1.00, 22.00, 2.00, 46.00, 69.00, 46.00,
  },
  {
    -41.00, -35.00, -45.00, -11.00, -1.00, -32.00,
  },
  {
    -35.00, -10.00, -35.00, 3.00, 22.00, 11.00,
  },
  {
    -44.00, 14.00, 22.00, -33.00, 22.00, 31.00,
  },
  {
    -32.00, -43.00, 3.00, 8.00, 5.00, 45.00,
  },
  {
    49.00, 3.00, 32.00, 68.00, 24.00, 62.00,
  },
  {
    -14.00, 10.00, -26.00, 34.00, 27.00, 22.00,
  },
  {
    -50.00, 30.00, 47.00, -8.00, 45.00, 59.00,
  },
  {
    4.00, 26.00, 22.00, 44.00, 44.00, 47.00,
  },
  {
    -41.00, -24.00, -5.00, -9.00, -21.00, 18.00,
  },
  {
    28.00, -30.00, 18.00, 67.00, -6.00, 29.00,
  },
  {
    -20.00, -28.00, -25.00, 27.00, 22.00, 17.00,
  },
  {
    -41.00, 46.00, -17.00, -25.00, 90.00, 28.00,
  },
  {
    21.00, 23.00, 39.00, 59.00, 66.00, 67.00,
  },
  {
    36.00, -2.00, -35.00, 80.00, 2.00, -19.00,
  },
  {
    37.00, -36.00, -41.00, 77.00, -9.00, 4.00,
  },
  {
    1.00, 36.00, -5.00, 21.00, 49.00, -2.00,
  },
  {
    -16.00, -12.00, 7.00, 18.00, 15.00, 21.00,
  },
  {
    -21.00, 15.00, -2.00, 0.00, 46.00, 41.00,
  },
  {
    33.00, -43.00, 4.00, 65.00, -34.00, 14.00,
  },
  {
    4.00, -2.00, -41.00, 14.00, 37.00, 0.00,
  },
  {
    1.00, -21.00, 5.00, 41.00, 24.00, 28.00,
  },
  {
    34.00, -21.00, 35.00, 73.00, -19.00, 62.00,
  },
}

local function test_region_ops_1 ()
	local combined = {}

	print ("region_ops_1: ")

	for _, aabb in ipairs (big_region) do
		table.insert (combined, aabb)
	end
	for _, aabb in ipairs (big_region_1) do
		table.insert (combined, aabb)
	end

	for i = 1, 1 do
		local rgn = mcl_util.decompose_AABBs (big_region)
		local rgn1 = mcl_util.decompose_AABBs (big_region_1)
		local dest = rgn:op (rgn1, mcl_util.OP_OR)
		local dest_s = dest:simplify ()
		local rgn_combined = mcl_util.decompose_AABBs (combined)
		assert (rgn_combined:equal_p (dest))
		assert (rgn_combined:equal_p (dest_s))
		rgn = dest:op (rgn1, mcl_util.OP_AND)
		assert (rgn:equal_p (rgn, rgn1))
		rgn = dest_s:op (rgn1, mcl_util.OP_AND)
		assert (rgn:equal_p (rgn, rgn1))
	end
end

local function collect_AABBs (aabb, data)
	table.insert (data, aabb)
	return nil
end

local function test_walk_AABBs ()
	local basic_cube = {
		-8.0, -8.0, -8.0,
		8.0, 8.0, 8.0,
	}
	local larger_cube = {
		-16.0, -16.0, -16.0,
		16.0, 16.0, 16.0,
	}
	local difference = {
		-- Vertical segments.
		{
			-16.0, -16.0, -16.0,
			16.0, -8.0, 16.0,
		},
		{
			-16.0, 8.0, -16.0,
			16.0, 16.0, 16.0,
		},
		-- Horizontal segments.
		{
			-16.0, -8.0, -16.0,
			-8.0, 8.0, 16.0,
		},
		{
			8.0, -8.0, -16.0,
			16.0, 8.0, 16.0,
		},
		-- Depthwise segments.
		{
			-8.0, -8.0, -16.0,
			8.0, 8.0, -8.0,
		},
		{
			-8.0, -8.0, 8.0,
			16.0, 16.0, 16.0,
		},
	}

	print ("test_walk_aabbs: ")
	local rgn = mcl_util.decompose_AABBs (big_region)
	local simple = rgn:simplify ()
	local aabbs = {}
	simple:walk (collect_AABBs, aabbs)
	rgn = mcl_util.decompose_AABBs (aabbs)
	print (string.format ("Yielded %d AABBs", #aabbs))
	assert (rgn:equal_p (simple))
	aabbs = {}
	rgn:walk (collect_AABBs, aabbs)
	simple = mcl_util.decompose_AABBs (aabbs)
	assert (rgn:equal_p (simple))

	aabbs = {}
	rgn = mcl_util.decompose_AABBs (big_region_1)
	simple = rgn:simplify ()
	simple:walk (collect_AABBs, aabbs)
	rgn = mcl_util.decompose_AABBs (aabbs)
	print (string.format ("Yielded %d AABBs", #aabbs))
	assert (rgn:equal_p (simple))
	aabbs = {}
	rgn:walk (collect_AABBs, aabbs)
	simple = mcl_util.decompose_AABBs (aabbs)
	assert (rgn:equal_p (simple))

	aabbs = {}
	simple = mcl_util.decompose_AABBs ({basic_cube})
	local larger = mcl_util.decompose_AABBs ({larger_cube})
	rgn = larger:op (simple, mcl_util.OP_SUB)
	rgn:walk (collect_AABBs, aabbs)
	rgn = mcl_util.decompose_AABBs (aabbs)
	print (string.format ("Yielded %d AABBs", #aabbs))
	simple = mcl_util.decompose_AABBs (difference)
	assert (simple:equal_p (rgn))
end

local function AABB_copy (octet, newx1, newy1, newz1, newx2, newy2, newz2)
	return {
		newx1 or octet[1],
		newy1 or octet[2],
		newz1 or octet[3],
		newx2 or octet[4],
		newy2 or octet[5],
		newz2 or octet[6],
	}
end

local function test_shorthand_ops ()
	print ("test_shorthand_ops: ")
	local basic_cube = {
		-8.0, -8.0, -8.0,
		8.0, 8.0, 8.0,
	};

	local big = mcl_util.decompose_AABBs (big_region_1)
	local lhs = mcl_util.decompose_AABBs ({basic_cube})
	local dest = big:subtract (basic_cube)
	local dest1 = big:op (lhs, mcl_util.OP_SUB)
	assert (dest1:equal_p (dest))

	big = mcl_util.decompose_AABBs (big_region_1)
	dest = big:intersect (basic_cube)
	dest1 = big:op (lhs, mcl_util.OP_AND)
	assert (dest1:equal_p (dest))

	big = mcl_util.decompose_AABBs (big_region_1)
	dest = big:union (basic_cube)
	dest1 = big:op (lhs, mcl_util.OP_OR)
	assert (dest1:equal_p (dest))
end

local function test_faces ()
	local OCTET_NPN = {
		0.0, 8.0, 0.0, 8.0, 16.0, 8.0,
	}
	local OCTET_NPP = {
		0.0, 8.0, 8.0, 8.0, 16.0, 16.0,
	}
	local OCTET_PPN = {
		8.0, 0.0, 8.0, 16.0, 8.0, 16.0,
	}
	local OCTET_PPP = {
		8.0, 8.0, 8.0, 16.0, 16.0, 16.0,
	}
	local BOTTOM_AABB = {
		0.0, 0.0, 0.0, 16.0, 8.0, 16.0,
	}

	local minecraft_stairs = {
		BOTTOM_AABB,
		OCTET_NPN,
		OCTET_PPN,
		OCTET_NPP,
		OCTET_PPP,
	}
	for _, item in ipairs (minecraft_stairs) do
		for i = 1, 6 do
			item[i] = item[i] - 8.0
		end
	end

	print ("test_faces: ")

	local stairs = mcl_util.decompose_AABBs (minecraft_stairs)
	-- Bottom face.
	local face = stairs:select_face ("y", -8.0)
	local test = mcl_util.decompose_AABBs ({{
		-8.0, -8.0, -8.0,
		8.0, 8.0, 8.0,
	}})
	assert (face:equal_p (test))

	-- Top face.
	face = stairs:select_face ("y", 8.0)
	test = mcl_util.decompose_AABBs ({
		AABB_copy (OCTET_PPP, nil, -8.0, nil, nil, nil, nil),
		AABB_copy (OCTET_NPN, nil, -8.0, nil, nil, nil, nil),
		AABB_copy (OCTET_NPP, nil, -8.0, nil, nil, nil, nil),
	})
	assert (face:equal_p (test))

	-- Left face.
	face = stairs:select_face ("x", -8.0)
	test = mcl_util.decompose_AABBs ({
		AABB_copy (OCTET_NPN, nil, nil, nil, 8.0, nil, nil),
		AABB_copy (OCTET_NPP, nil, nil, nil, 8.0, nil, nil),
		AABB_copy (BOTTOM_AABB, nil, nil, nil, 8.0, nil, nil),
	})
	assert (face:equal_p (test))

	-- Right face.
	face = stairs:select_face ("x", 8.0)
	test = mcl_util.decompose_AABBs ({
		AABB_copy (OCTET_PPP, -8.0, nil, nil, nil, nil, nil),
		AABB_copy (OCTET_PPN, -8.0, nil, nil, nil, nil, nil),
		AABB_copy (BOTTOM_AABB, -8.0, nil, nil, nil, nil, nil),
	})
	assert (face:equal_p (test))

	-- Front face.
	face = stairs:select_face ("z", -8.0)
	test = mcl_util.decompose_AABBs ({
		AABB_copy (OCTET_NPN, nil, nil, nil, nil, nil, 8.0),
		AABB_copy (BOTTOM_AABB, nil, nil, nil, nil, nil, 8.0),
	})
	assert (face:equal_p (test))

	-- Rear face.
	face = stairs:select_face ("z", 8.0)
	test = mcl_util.decompose_AABBs ({{
		-8.0, -8.0, -8.0,
		8.0, 8.0, 8.0,
	}})
	assert (face:equal_p (test))

	local fence_post = {
		-0.125, -0.5, -0.125,
		0.125, 1.0, 0.125,
	};
	local fence_post_wanted = {
		-0.125, -0.5, -0.125,
		0.125, 0.5, 0.125,
	};

	stairs = mcl_util.decompose_AABBs ({fence_post})
	face = stairs:select_face ("y", 0.5)
	test = mcl_util.decompose_AABBs ({fence_post_wanted})
	assert (test:equal_p (face))
end

local function test_empty_region ()
	print ("test_empty_region: ")

	local rgn2 = mcl_util.decompose_AABBs (big_region_1)
	local empty = mcl_util.empty_region
	local rgn1 = empty:op (rgn2, mcl_util.OP_OR)
	assert (rgn1:equal_p (rgn2))
	local rgn1 = empty:op (rgn2, mcl_util.OP_AND)
	assert (rgn1:equal_p (empty) and rgn1:is_empty ())
	local rgn1 = empty:op (rgn2, mcl_util.OP_SUB)
	assert (rgn1:equal_p (empty) and rgn1:is_empty ())
	local rgn1 = rgn2:op (empty, mcl_util.OP_SUB)
	assert (rgn1:equal_p (rgn2))
	local rgn1 = empty:op (rgn2, mcl_util.OP_NEQ)
	assert (rgn1:equal_p (rgn2))
end

test_AABB_to_rgn ()
test_AABBs_to_rgn ()
test_region_ops ()
test_region_ops_1 ()
test_walk_AABBs ()
test_shorthand_ops ()
test_faces ()
test_empty_region ()
